<?php

namespace Modules\Appointment\Repositories\Eloquents;

use App\Repositories\Eloquents\BaseRepository;
use App\Traits\SendNotification;
use App\User;
use Brian2694\Toastr\Facades\Toastr;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Illuminate\Database\Eloquent\Model;
use Modules\Appointment\Entities\Booking;
use Modules\Appointment\Entities\BookTrailLesson;
use Modules\Appointment\Entities\InstructorTeachingCategory;
use Modules\Appointment\Entities\Schedule;
use Modules\Appointment\Entities\TimeSlot;
use Modules\Appointment\Repositories\Interfaces\InstructorRepositoryInterface;
use Modules\Appointment\Repositories\Interfaces\ScheduleRepositoryInterface;
use Modules\Appointment\Repositories\Interfaces\TimeSlotRepositoryInterface;
use Modules\CourseSetting\Entities\Category;
use phpDocumentor\Reflection\Types\Nullable;

class ScheduleRepository extends BaseRepository implements ScheduleRepositoryInterface
{
    use SendNotification;

    protected $user;
    protected $booking;
    protected $category;
    protected $timeSlotRepository;
    protected $instructorRepository;
    protected $instructorTeachingCategory;

    public function __construct(
        User                          $user,
        Schedule                      $model,
        Booking                       $booking,
        Category                      $category,
        TimeSlot                      $timeSlot,
        TimeSlotRepositoryInterface   $timeSlotRepository,
        InstructorRepositoryInterface $instructorRepository,
        InstructorTeachingCategory    $instructorTeachingCategory
    )
    {
        parent::__construct($model);
        $this->user = $user;
        $this->booking = $booking;
        $this->category = $category;
        $this->timeSlot = $timeSlot;
        $this->timeSlotRepository = $timeSlotRepository;
        $this->instructorRepository = $instructorRepository;
        $this->instructorTeachingCategory = $instructorTeachingCategory;
    }

    public function index($next_date = null, $pre_date = null): array
    {
        $data = [];
        $data += $this->instructorRepository->instructors();
        $data += $this->datePeriods($next_date, $pre_date);
        $data += $this->instructorSchedule();
        return $data;
    }

    public function changeWeek(array $payload): array
    {
        $next_date = gv($payload, 'next_date');
        $pre_date = gv($payload, 'pre_date');

        $data = [];
        $data += $this->instructorRepository->instructors();
        $data += $this->datePeriods($next_date, $pre_date);
        $data += $this->instructorSchedule(gv($payload, 'instructor'));
        return $data;
    }

    public function create(array $payload): ?Model
    {

        $start_date = date('Y-m-d', strtotime(gv($payload, 'start_date')));
        $end_date = date('Y-m-d', strtotime(gv($payload, 'end_date')));

        if (gv($payload, 'end_date')) {
            $dates = CarbonPeriod::create($start_date, $end_date);
            foreach ($dates as $date) {
                $payload['date'] = $date->format('Y-m-d');
                $this->model->create($this->formatParams($payload));
            }
            return null;
        } else {
            $payload['date'] = date('Y-m-d', strtotime(gv($payload, 'start_date')));

            return $this->model->create($this->formatParams($payload));
        }
    }

    public function update(int $modelId, array $payload): bool
    {

        $model = $this->findById($modelId);
        $date = date('Y-m-d', strtotime(gv($payload, 'date')));
        $checkSchedule = $this->model->where('schedule_date', $date)
            ->where('id', '!=', $modelId)
            ->where('slot_id', $payload['slot'])
            ->where('user_id', gv($payload, 'user'))
            ->whereIn('status', [1])->first();
        if ($checkSchedule) {
            return false;
        }
        $update = $model->update($this->formatParams($payload, $modelId));
        if ($update) {
            $timeSlotModel = $this->timeSlot->where('id', $payload['slot'])->first();
            $start_time = $timeSlotModel->start_time;
            $end_time = $timeSlotModel->end_time;
            $timeSlot = $start_time . '-' . $end_time;
             if (gv($payload, 'move_slot') == 1) {
                if (auth()->user()->id != gv($payload, 'user')) {
                    $instructor = $this->user->where('id', gv($payload, 'user'))->first();
                    $shortCodes = [
                        'date' => showDate($date),
                        'timeSlot' => $timeSlot,
                        'instructor' => $instructor->name,
                    ];
                    $this->sendNotification('Schedule_Move_Admin', $instructor, $shortCodes);


                }
                // for student
                if ($modelId) {
                    $students = $this->booking->with('userInfo')->where('schedule_id', $modelId)->get();
                    foreach ($students as $student) {
                        $this->sendNotification('Schedule_Move_Student', $student->userInfo, $shortCodes);
                    }

                }
            }
        }

        return $update;

    }

    public function edit($modelId): array
    {
        $data['editSchedule'] = $this->findById($modelId);
        return $data;
    }

    private function formatParams($payload, $modelId = null)
    {
        $dayDate = date('m/d/Y', strtotime(gv($payload, 'date')));
        $day = Carbon::createFromFormat('m/d/Y', $dayDate)->format('l');
        $formatParam = [
            'schedule_date' => date('Y-m-d', strtotime(gv($payload, 'date'))),
            'day' => strtolower($day),
            'user_id' => gv($payload, 'user'),
            'category_id' => gv($payload, 'category'),
            'sub_category_id' => gv($payload, 'sub_category'),
            'updated_by' => auth()->user()->id,

        ];

        if ($modelId && gv($payload, 'move_slot') == 1) {
            $formatParam['slot_id'] = gv($payload, 'slot');
        } elseif (!$modelId) {
            $formatParam['slot_id'] = gv($payload, 'slot');
        }

        if (!$modelId) {
            $formatParam['created_by'] = auth()->user()->id;
        }

        return $formatParam;
    }

    public function datePeriods($next_date = null, $pre_date = null): array
    {
        if ($next_date) {
            $data = $this->nextWeek($next_date);
        } elseif ($pre_date) {
            $data = $this->preWeek($pre_date);
        } else {
            $data['this_week'] = $weekNumber = date("W");
            $data['periods'] = CarbonPeriod::create(Carbon::now()
                ->startOfWeek(Carbon::SUNDAY)->format('Y-m-d'), Carbon::now()
                ->endOfWeek(Carbon::SATURDAY)->format('Y-m-d'));
        }
        $data['weekDates'] = [];

        foreach ($data['periods'] as $date) {
            $data['weekDates'][] =  Carbon::parse($date)->locale('en')->translatedFormat(Settings('active_date_format'));
        }
        return $data;
    }

    public function searchSchedule(array $payload): array
    {
        $user = $this->instructorRepository->instructorById(gv($payload, 'instructor'));
        return $user;
    }

    public function schedule($slug): array
    {

        $data = $this->instructorRepository->instructorBySlug($slug);
        $data += $this->instructorSchedule($data['instructor']['id']);
        $data += $this->index(null, null, $data['instructor']['id']);
        return $data;
    }

    public function instructorSchedule($id = null): array
    {

        $schedule = $this->model->when(auth()->user()->role_id == 1 && $id, function ($query) use ($id) {
            $query->where('user_id', $id);
        }, function ($query) {
            $query->where('user_id', auth()->user()->id);
        });

        $teachingCategoryIds = $this->instructorTeachingCategory
            ->when($id && auth()->user()->role_id == 1, function ($query) use ($id) {
                $query->where('instructor_id', $id);
            }, function ($query) {
                $query->when(auth()->user()->role_id != 1, function ($subQuery) {
                    $subQuery->where('instructor_id', auth()->user()->id);
                });
            })->pluck('category_id')->toArray();

        $data['schedules'] = $schedule->get();
        $data['user_id'] = $id ?? auth()->user()->id;
        $data['hasSchedule'] = $schedule->pluck('schedule_date')->toArray();
        $data['hasScheduleSlot'] = $schedule->groupBy('slot_id')->pluck('slot_id')->toArray();
        $data['teachingCategoryIds'] = $teachingCategoryIds;

        $data['categories'] = $this->category->get(['id', 'name']);

        $data['timeSlots'] = $this->timeSlot->when(auth()->user()->role_id == 1 && $id, function ($query) use ($id) {
            $query->where('user_id', $id);
        }, function ($query) {
            $query->where('user_id', auth()->user()->id);
        })->where('status', 1)->get();

        return $data;
    }

    private function nextWeek($next_date): array
    {
        $start_date = Carbon::parse($next_date)->addDay(1);
        $date = Carbon::parse($next_date);

        $end_date = Carbon::parse($start_date)->addDay(6);

        $data['this_week'] = $week_number = $end_date->weekOfYear;

        $data['periods'] = CarbonPeriod::create($start_date, $end_date);
        return $data;
    }

    private function preWeek($pre_date): array
    {
        $end_date = Carbon::parse($pre_date)->subDays(1);

        $start_date = Carbon::parse($end_date)->subDays(6);

        $data['this_week'] = $week_number = $end_date->weekOfYear;

        $data['periods'] = CarbonPeriod::create($start_date, $end_date);
        return $data;
    }

    public function userSchedule(array $payload)
    {
        $today = Carbon::now()->format('Y-m-d');
        return $this->model->where('schedule_date', '>', $today)->groupBy('schedule_date')
            ->where('user_id', gv($payload, 'instructor_id'))->get();
    }

    public function userTimeSlot(array $payload)
    {
        $date = $this->model->where('id', gv($payload, 'schedule_date'))->value('schedule_date');
        return $this->model->whereDate('schedule_date', $date)
            ->where('user_id', gv($payload, 'instructor_id'))->get()->map(function ($item, $key) {
                return [
                    'id' => $item->id,
                    'startToEnd' => date('h:i A', strtotime(@$item->slotInfo->start_time)) . '-' . date('h:i A', strtotime(@$item->slotInfo->end_time)),
                ];
            });
    }

    public function setSchedule(array $payload)
    {

        $trail = BookTrailLesson::find(gv($payload, 'trail_id'));
        $scheduleBooking = new Booking;
        $scheduleBooking->user_id = $trail->student_id;
        $scheduleBooking->instructor_id = gv($payload, 'schedule_instructor');
        $scheduleBooking->schedule_id = gv($payload, 'time_slot');
        $scheduleBooking->tracking = getTrx(20);
        $scheduleBooking->purchase_price = 0;
        $scheduleBooking->coupon = null;
        $scheduleBooking->status = 1;
        $scheduleBooking->save();
        $trail->update([
            'schedule_id' => $scheduleBooking->schedule_id,
            'status' => 0
        ]);
        $start_time = date('h:i A', strtotime($scheduleBooking->schedule->slotInfo->start_time));
        $end_time = date('h:i A', strtotime($scheduleBooking->schedule->slotInfo->end_time));
        $timeSlot = $start_time . '-' . $end_time;
        $shortCodes = [
            'date' => $scheduleBooking->schedule->schedule_date,
            'timeSlot' => $timeSlot,
            'instructor' => $trail->instructorInfo->name,
        ];

        $this->sendNotification('Book_Trail_Lesson_Approve', $trail->userInfo, $shortCodes);

    }

    public function deleteAble($id)
    {
        $model = $this->booking->where('schedule_id', $id)->first();
        if ($model) {
            Toastr::warning(trans('appointment.Student Booked This Schedule'), trans('common.Warning'));
            return null;
        }
        return $this->findById($id);
    }

    public function deleteById(int $modelId): bool
    {
        $model = $this->deleteAble($modelId);
        if ($model) {
            return $model->delete();
        }
        return false;
    }
}
