<?php

namespace Modules\WhatsappSupport\Http\Controllers;

use App\Traits\UploadMedia;
use Brian2694\Toastr\Facades\Toastr;
use Carbon\Carbon;
use Illuminate\Routing\Controller;
use Modules\WhatsappSupport\Entities\Agents;
use Modules\WhatsappSupport\Http\Requests\AgentCreation;

class AgentController extends Controller
{
    use UploadMedia;

    public function index()
    {
        $agents = Agents::all();
        return view('whatsappsupport::agents.index', compact('agents'));
    }

    public function store(AgentCreation $request)
    {
         if (demoCheck()) {
            return redirect()->back();
        }
        if (!$request->always_available) {
            if (!$request->has('day')) {
                session()->flash('error', 'Please select/check at least one working day!');
                return redirect()->back()->withInput();
            }
        }

        $agent = Agents::create([
            'name' => $request->name,
            'designation' => $request->designation,
            'number' => $request->number??'',
            'always_available' => (int)$request->always_available,
            'status' => (int)$request->always_available,
            'type'=>(int)$request->type??1,
            'url'=>$request->url??''
        ]);

        if ($request->avatar) {
            $agent->avatar = $this->generateLink($request->avatar, $agent->id, get_class($agent), 'avatar');
        }
        $agent->save();
        if (!$request->always_available) {
            foreach ($request->day as $day) {
                $index = $this->getDayIndex($day);
                $agent->times()->updateOrCreate(
                    [
                        'agent_id' => $agent->id,
                        'day' => $day
                    ], [
                    'day' => $day,
                    'start' => Carbon::parse($request->start[$index])->format('H:i:s'),
                    'end' => Carbon::parse($request->end[$index])->format('H:i:s')
                ]);
            }
        }

        Toastr::success(trans('common.Operation successful'), trans('common.Success'));

        return redirect()->route('whatsapp-support.agents');

    }

    public function create()
    {
        return view('whatsappsupport::agents.create');
    }

    public function getDayIndex($day)
    {
        switch ($day) {
            case 'Sunday' :
                return 0;
            case 'Monday' :
                return 1;
            case 'Tuesday' :
                return 2;
            case 'Wednesday' :
                return 3;
            case 'Thursday' :
                return 4;
            case 'Friday' :
                return 5;
            case 'Saturday' :
                return 6;
        }
    }

    public function show($id)
    {
        $agent = Agents::find($id);
        return view('whatsappsupport::agents.show', compact('agent'));
    }

    public function edit($id)
    {
        return view('whatsappsupport::edit');
    }

    public function update(AgentCreation $request)
    {
        if (demoCheck()) {
            return redirect()->back();
        }
        $agent = Agents::find($request->agent_id);

        if (!$request->always_available) {
            if (!$request->has('day')) {
                session()->flash('error', 'Please select/check at least one working day!');
                return redirect()->back();
            }
        }


        $agent->update([
            'name' => $request->name,
            'avatar' => '',
            'designation' => $request->designation,
            'number' => $request->number??"",
            'always_available' => (int)$request->always_available,
            'status' => (int)$request->status,
            'type'=>(int)$request->type??1,
            'url'=>$request->url??''
        ]);

        $this->removeLink($agent->id, get_class($agent));
        if ($request->avatar) {
            $agent->avatar = $this->generateLink($request->avatar, $agent->id, get_class($agent), 'avatar');
        }
        $agent->save();
        if (!$request->always_available) {
            $agent->times()->delete();
            foreach ($request->day as $day) {
                $index = $this->getDayIndex($day);
                $agent->times()->updateOrCreate(
                    [
                        'agent_id' => $agent->id,
                        'day' => $day
                    ], [
                    'day' => $day,
                    'start' => Carbon::parse($request->start[$index])->format('H:i:s'),
                    'end' => Carbon::parse($request->end[$index])->format('H:i:s')
                ]);
            }
        }

        Toastr::success(trans('common.Operation successful'), trans('common.Success'));

        return redirect()->route('whatsapp-support.agents');
    }

    public function destroy($id)
    {
        $agent = Agents::find($id);
        $agent->times()->delete();
        $agent->delete();
        Toastr::success(trans('common.Operation successful'), trans('common.Success'));
        return redirect()->route('whatsapp-support.agents');

    }
}
