<?php

namespace Modules\Store\Http\Controllers;

use App\User;
use Brian2694\Toastr\Facades\Toastr;
use Exception;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use Modules\CourseSetting\Entities\Course;
use Modules\Store\Entities\Product;
use Modules\Store\Entities\ProductAttribute;
use Modules\Store\Entities\ProductAuthor;
use Modules\Store\Entities\ProductBrand;
use Modules\Store\Entities\ProductCategory;
use Modules\Store\Entities\ProductUnit;
use Modules\Store\Exports\ProductExport;
use Modules\Store\Exports\ProductSampleExport;
use Modules\Store\Repositories\ProductRepository;
use Yajra\DataTables\Facades\DataTables;

class ProductController extends Controller
{

    use ValidatesRequests;

    public $ProductRepository;

    public function __construct(ProductRepository $ProductRepository)
    {
        $this->ProductRepository = $ProductRepository;
    }


    public function index()
    {

        try {
            $products = [];
            $query = ProductCategory::orderBy('position_order', 'ASC');
            $categories = $query->with('parent')->get();
            $users = User::where('status', 1)->whereIn('role_id', [1, 2])->select('id', 'name')->get();
            $search_user = !empty(request('users')) ? request('users') : '';
            $search_productType = !empty(request('productType')) ? request('productType') : '';
            $product_status = !empty(request('product_status')) ? request('product_status') : '';
            $req_category = !empty(request('category')) ? request('category') : '';

            return view('store::products.index', compact('products', 'categories', 'users', 'search_user', 'search_productType', 'product_status', 'req_category'));
        } catch (Exception $e) {
            Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
            return redirect()->back();
        }
    }

    public function getAllproducts(Request $request)
    {
        $query = Product::query();

        if (Auth::user()->role_id == 1) {
            $query->with('category')->get();
        } else {
            $query->with('category')->where('user_id', Auth::user()->id)->get();
        }
        if (!empty(request('category'))) {
            $query->where('category_id', request('category'));
        }
        if (!empty(request('productType'))) {
            $query->where('type', request('productType'));
        }
        if (!empty(request('users'))) {
            $query->where('user_id', request('users'));
        }
        if ($request->product_status != "") {
            if ($request->product_status == 'active') {
                $query->where('status', 1);
            } elseif ($request->product_status == 'inactive') {
                $query->where('status', 0);
            }
        }
        $query->latest();

        return Datatables::of($query)
            ->addIndexColumn()
            ->editColumn('title', function ($query) {
                return $query->title;
            })
            ->addColumn('type', function ($query) {
                return $query->type == 1 ? trans('product.Virtual') : trans('product.Physical');
            })

            ->editColumn('category', function ($query) {
                if ($query->category) {
                    return $query->category->title;
                } else {
                    return '';
                }
            })
            ->addColumn('action', function ($query) {
                return view('store::partials._td_action', compact('query'));
            })
            ->addColumn('status', function ($query) {
                $status_enable_eisable = "";
                if (permissionCheck('store.product.status')) {
                    $status_enable_eisable = "status_enable_disable";
                }
                $checked = $query->course->status == 1 ? "checked" : "";
                return '<label class="switch_toggle"  >
                                                    <input type="checkbox" class="' . $status_enable_eisable . '"
                                                             value="' . $query->course->id . '"
                                                             ' . $checked . '><i class="slider round"></i></label>';


            })
            ->rawColumns(['status', 'action'])->make(true);
    }

    public function ajaxGetProductSubCategory(Request $request)
    {

        try {

            $query = ProductCategory::orderBy('position_order', 'ASC')->where('status', 1);
            $sub_categories = $query->where('parent_id', '=', $request->id)->get();
            return $sub_categories;
        } catch (Exception $e) {
            return response()->json("", 404);
        }
    }

    public function store(Request $request)
    {
        $code = auth()->user()->language_code;

        $rules = [
            'name.' . $code => 'required|max:255',
            'category' => 'required',
            'price' => 'nullable',
        ];

        if ($request->productType == 1) {
            $rules['soft_file'] = 'required';
        }


        $this->validate($request, $rules, validationMessage($rules));

        if ($request->get('productOption',0) != 1 && $request->discount) {
            if ($request->discount_type == 1) {
                $dis_price = $request->price - $request->discount;
            } else {
                $dis_price = $request->price - $request->price * $request->discount / 100;
            }
            if ($dis_price < 0) {
                Toastr::error(trans('product.Invalid Discount'), trans('common.Failed'));
                return redirect()->back();
            }
        }

        try {
            $product = $this->ProductRepository->create($request);
            if ($product) {
                Toastr::success(trans('common.Operation successful'), trans('common.Success'));
                return redirect()->route('store.product.index');
            } else {
                Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
                return redirect()->back();
            }
        } catch (Exception $e) {
            GettingError($e->getMessage(), url()->current(), request()->ip(), request()->userAgent());
        }
    }

    public function create()

    {
        $query = ProductCategory::orderBy('position_order', 'ASC')->where('status', 1);
        $categories = $query->with('parent')->get();
        $authors = ProductAuthor::where('status', 1)->get();
        $brands = ProductBrand::where('status', 1)->get();
        $units = ProductUnit::where('status', 1)->get();
        $attributes = ProductAttribute::with('values')->get();
        return view('store::products.create', compact('authors', 'categories', 'brands', 'attributes', 'units'));
    }

    public function edit($id)
    {
        $user = Auth::user();
        $product = Product::with('variations')->findOrFail($id);
        $query = ProductCategory::orderBy('position_order', 'ASC')->where('status', 1);
        $categories = $query->with('parent')->get();
        $authors = ProductAuthor::where('status', 1)->get();
        $brands = ProductBrand::where('status', 1)->get();
        $units = ProductUnit::where('status', 1)->get();
        $attributes = ProductAttribute::with('values')->get();

        return view('store::products.create', compact('product', 'authors', 'categories', 'brands', 'attributes', 'units'));
    }

    public function update(Request $request)
    {

        $code = auth()->user()->language_code;

        $rules = [
            'name.' . $code => 'required|max:255',
            'category' => 'required',
            'price' => 'nullable',
        ];
        if ($request->productType == 1) {
            $rules['soft_file'] = 'required';
        }
        if ($request->discount > 0) {
            $rules['discount_type'] = 'required';
        }

        $this->validate($request, $rules, validationMessage($rules));

        if ($request->get('productOption',0) != 1 && $request->discount) {
            if ($request->discount_type == 1) {
                $dis_price = $request->price - $request->discount;
            } else {
                $dis_price = $request->price - $request->price * $request->discount / 100;
            }
            if ($dis_price < 0) {
                Toastr::error(trans('product.Invalid Discount'), trans('common.Failed'));
                return redirect()->back();
            }
        }

        try {
            $product = $this->ProductRepository->update($request);

            if ($product) {
                Toastr::success(trans('common.Operation successful'), trans('common.Success'));
                return redirect()->route('store.product.index');
            } else {
                Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
                return redirect()->back();
            }
        } catch (Exception $e) {
            GettingError($e->getMessage(), url()->current(), request()->ip(), request()->userAgent());
        }
    }

    public function destroy($id)
    {
        if (demoCheck()) {
            return redirect()->back();
        }
        $product = Product::find($id);
        if ($product){
            Course::where('product_id', $id)->delete();
            $product->delete();
        }
         Toastr::success(trans('common.Operation successful'), trans('common.Success'));
        return redirect()->route('store.product.index');

    }

    public function exportAllProduct()
    {
        $query = Product::orderBy('id', 'desc');
        if (!empty(request('category'))) {
            $query->where('category_id', request('category'));
        }
        if (!empty(request('productType'))) {
            $query->where('type', request('productType'));
        }
        if (!empty(request('users'))) {
            $query->where('user_id', request('users'));
        }
        if (request()->product_status != "") {
            if (request()->product_status == 'active') {
                $query->where('status', 1);
            } elseif (request()->product_status == 'inactive') {
                $query->where('status', 0);
            }
        }
        $data = $query->get();
        return Excel::download(new ProductExport($data), 'all-products.xlsx');
    }

    public function exportSampleProduct()
    {
        return Excel::download(new ProductSampleExport(), 'sample-product.xlsx');
    }

    public function sku_combination(Request $request)
    {
        $options = array();
        foreach ($request->get('attributes', []) as $key => $no) {
            $name = 'attribute_values_' . $no;
            $data = array();
            if (isset($request[$name])) {
                foreach ($request[$name] as $key => $item) {
                    array_push($data, $item);
                }
            }
            array_push($options, $data);
        }
        $attributes = $request->get('attributes', []);
        $combinations = combinations($options);

        $selling_price_sku = !empty($request->old_sku_price) ? explode(',', $request->old_sku_price) : [];
        $sku_stock = !empty($request->old_sku_stock) ? explode(',', $request->old_sku_stock) : [];
        $old_sku = !empty($request->old_sku) ? explode(',', $request->old_sku) : [];
        return view('store::products.sku_combinations', compact('combinations', 'attributes', 'selling_price_sku', 'sku_stock', 'old_sku'));
    }

    public function show($id)
    {
        $product = Product::with(['category', 'subCategory', 'brand', 'user','course'])->with('variations')->findOrFail($id);
        return view('store::products.show',compact('product'));
    }
}
