<?php

namespace Modules\Store\Http\Controllers;

use App\User;
use Brian2694\Toastr\Facades\Toastr;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Modules\Payment\Entities\Checkout;
use Modules\Setting\Repositories\GeneralSettingRepositoryInterface;
use Modules\Store\Entities\CancelReason;
use Modules\Store\Entities\DeliveryProcess;
use Modules\Store\Entities\OrderPackageDetail;
use Modules\Store\Entities\ProductCarrier;
use Modules\Store\Repositories\OrderRepository;
use Yajra\DataTables\Facades\DataTables;

class ProductOrderController extends Controller
{
    protected $orderRepository;
    protected $generalSettingRepository;

    public function __construct(OrderRepository $orderRepository, GeneralSettingRepositoryInterface $generalSettingRepository)
    {
        $this->orderRepository = $orderRepository;
        $this->generalSettingRepository = $generalSettingRepository;
    }

    public function index(Request $request)
    {
        try {
            $data['title'] = _trans('store.Order List');
            $data['orders'] = $this->orderRepository->getAll();
            $data['carriers'] = ProductCarrier::where('status', 1)->get();
            return view('store::order.index', compact('data'));
        } catch (Exception $e) {
            GettingError($e->getMessage(), url()->current(), request()->ip(), request()->userAgent());
        }
    }

    public function getAllOrders(Request $request)
    {
        $query = Checkout::with('courses.course');
        if (Auth::user()->role_id != 1) {
            $query->whereHas('packages', function ($q) {
                $q->where('seller_id', Auth::user()->id);
            });
        }
        if (!empty(request('order_status'))) {
            $query->where('is_confirmed', request('order_status'));
        }

        $query->whereHas('courses', function ($q) {
            $q->where('is_store', 1);
        })->latest();

        return Datatables::of($query)
            ->addIndexColumn()
            ->addColumn('date', function ($query) {
                return showDate(@$query->created_at);
            })
            ->editColumn('order_number', function ($query) {
                return $query->order_number;
            })
            ->addColumn('customer', function ($query) {
                return @$query->user->name;
            })
            ->addColumn('instructor', function ($query) {
                $instructor = [];
                foreach ($query->courses as $item) {
                    $instructor[] = $item->course->user->name;
                }
                return @$instructor;
            })
            ->addColumn('quantity', function ($query) {
                return $query->courses->sum('qty');
            })
            ->editColumn('price', function ($query) {
                return view('store::order.partials._course_price_td', compact('query'));
            })
            ->addColumn('discount', function ($query) {
                return $query->discount == 0 ? 0 : getPriceFormat($query->discount);
            })
            ->addColumn('action', function ($query) {
                return view('store::order.partials._td_action', compact('query'));
            })
            ->addColumn('status', function ($query) {
                $route = 'store.product.status';
                return view('store::order.partials._td_status', compact('query', 'route'));
            })
            ->addColumn('is_paid', function ($query) {
                return view('store::order.partials._is_paid_td', compact('query'));
            })
            ->rawColumns(['status', 'action'])->make(true);
    }

    public function show($id)
    {
        $data['order'] = Checkout::with('bill')->find($id);

        if (Auth::user()->role_id != 1) {
            $data['packages'] = OrderPackageDetail::with('product_details.getCourse', 'products.course', 'products')->whereHas('product_details', function ($q) {
                $q->where('seller_id', Auth::user()->id);
            })->where('order_id', $id)->groupby('seller_id')->distinct()->get();
        } else {
            $data['packages'] = OrderPackageDetail::with('product_details.getCourse', 'products')->where('order_id', $id)->groupby('seller_id')->distinct()->get();
        }
        $data['processes'] = DeliveryProcess::all();
        $data['cancel_reasons'] = CancelReason::all();
        return view('store::order.sale_details', $data);
    }

    public function globalPrint($id)
    {
        $data['order'] = Checkout::with('bill')->find($id);
        if (Auth::user()->role_id != 1) {
            $data['packages'] = OrderPackageDetail::with('product_details.getCourse', 'products.course')->whereHas('product_details', function ($q) {
                $q->where('seller_id', Auth::user()->id);
            })->where('order_id', $id)->groupby('seller_id')->distinct()->get();
        } else {
            $data['packages'] = OrderPackageDetail::with('product_details.getCourse')->where('order_id', $id)->groupby('seller_id')->distinct()->get();
        }
        return view('store::order.sale_details_print', $data);
    }

    public function myGlobalPrint($id)
    {
        $data['order'] = Checkout::with('bill')->find($id);
        if (Auth::user()->role_id != 1) {
            $data['packages'] = OrderPackageDetail::with('product_details.getCourse')->where(['order_id' => $id, 'seller_id' => auth()->user()->id])->get();
        } else {
            $data['packages'] = OrderPackageDetail::with('product_details.getCourse')->where(['order_id' => $id, 'seller_id' => auth()->user()->id])->get();
        }
        return view('store::order.my_sale_details_print', $data);
    }

    public function orderConfirm($id)
    {
        try {
            $result = Checkout::find($id);
            $result->is_confirmed = 1;
            $result->save();
            Toastr::success(trans('common.Operation successful'), trans('common.Success'));
            return redirect()->back();
        } catch (Exception $e) {
            Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
            return back();
        }
    }

    public function track_order_configuration()
    {
        try {
            return view('store::order.track_order_configuration');
        } catch (Exception $e) {
            Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
            return back();
        }
    }

    public function sales_info_update(Request $request, $id)
    {
        $data = $request->all();
        unset($data['cancel_reason_id']);
        unset($data['_token']);
        if (!empty($request->cancel_reason_id)) {
            $data['cancel_reason_id'] = $request->cancel_reason_id;
        }

        try {
            $data['order'] = $this->orderRepository->orderInfoUpdate($data, $id);
            Toastr::success(trans('common.Operation successful'), trans('common.Success'));
            return back();
        } catch (Exception $e) {
            Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
            return back();
        }
    }

    public function track_order_configuration_update(Request $request)
    {

        if ($request->track_order_by_secret_id) {
            $track_order_by_secret_id = 1;
        } else {
            $track_order_by_secret_id = 0;
        }
        if ($request->cash_on_delivery) {
            $cash_on_delivery = 1;
        } else {
            $cash_on_delivery = 0;
        }
        $data = [
            'track_order_by_secret_id' => (int)$track_order_by_secret_id,
            'cash_on_delivery' => (int)$cash_on_delivery,
        ];

        $this->generalSettingRepository->update($data);
        Toastr::success(trans('common.Operation successful'), trans('common.Success'));
        return redirect()->back();
    }

    public function my_sales_index()
    {

        try {
            $data['title'] = _trans('store. My Order List');
            return view('store::order.my_order_index', compact('data'));
        } catch (Exception $e) {
            GettingError($e->getMessage(), url()->current(), request()->ip(), request()->userAgent());
        }

    }

    public function my_sales_get_data()
    {
        $query = OrderPackageDetail::with('product_details.getCourse', 'order')
            ->where('seller_id', Auth::id())->groupby('order_id')->distinct();
        $query->latest();

        return Datatables::of($query)
            ->addIndexColumn()
            ->addColumn('date', function ($query) {
                return showDate($query->updated_at);
            })
            ->editColumn('order_number', function ($query) {
                return @$query->order->order_number;
            })
            ->addColumn('customer', function ($query) {
                return @$query->order->user->email;
            })
            ->addColumn('quantity', function ($query) {
                return 0;
            })
            ->addColumn('discount', function ($query) {
                return $query->discount == 0 ? 0 : getPriceFormat($query->discount);
            })
            ->addColumn('action', function ($query) {
                return view('store::order.partials._my_td_action', compact('query'));
            })
            ->addColumn('status', function ($query) {
                return view('store::order.partials._my_td_status', compact('query'));
            })
            ->addColumn('order_state', function ($query) {
                return view('store::order.partials._my_order_order_state_td', compact('query'));
            })
            ->rawColumns(['status', 'action'])->make(true);
    }

    public function my_sale_show($id)
    {
        $data['package'] = OrderPackageDetail::with('products.course')->find($id);
        $data['order'] = Checkout::with('billing')->find($data['package']->order_id);
        $data['processes'] = DeliveryProcess::all();
        return view('store::order.my_sale_details', $data);
    }

    public function update_delivery(Request $request, $id)
    {
        try {
            $data['order'] = $this->orderRepository->updateDeliveryStatus($request->except("_token"), $id);
            Toastr::success(trans('common.Operation successful'), trans('common.Success'));
            return back();
        } catch (Exception $e) {
            Toastr::error(__('common.error_message'));
            return back();
        }
    }

    public function my_purchase_order_package_cancel(Request $request)
    {
        try {
            $data = OrderPackageDetail::find($request->order_id);
            if ($data->order->is_cancelled == 0) {
                $data->order->is_cancelled = 1;
                $data->order->cancel_reason_id = $request->reason;
                $data->order->save();
            }
            $data->is_cancelled = 1;
            $data->cancel_reason_id = $request->reason;
            $data->save();

            foreach($data->order->courses as $course){
                $course->status = 0;
                $course->save();
            }
            if ($data->order->is_paid == 1) {
                $user = User::find($data->order->user_id);
                if ($user){
                    $user->balance = $user->balance + $data->order->purchase_price;
                    $user->save();
                }

            }

            Toastr::success(trans('common.Operation successful'), trans('common.Success'));
            return back();
        } catch (Exception $e) {
            Toastr::error(__('order.your_order_cancellation_has_been_failed_try_again'));
            return back();
        }
    }

}
